document.addEventListener('DOMContentLoaded', () => {
    // Get DOM elements
    const htmlInput = document.getElementById('html-input');
    const cssInput = document.getElementById('css-input');
    const jsInput = document.getElementById('js-input');
    // const fileStructure = document.getElementById('file-structure'); // Removed
    const tabButtons = document.querySelectorAll('.tab-btn');
    const inputSections = document.querySelectorAll('.input-section');
    const previewBtn = document.getElementById('preview-btn');
    const previewNewTabBtn = document.getElementById('preview-new-tab-btn');
    const generateBtn = document.getElementById('generate-zip');
    const clearBtn = document.getElementById('clear-btn');
    const previewSection = document.getElementById('preview-section');
    const previewContainer = document.getElementById('preview-container');
    const resultSection = document.getElementById('result-section');
    const filenameDisplay = document.getElementById('filename-display');
    const downloadLink = document.getElementById('download-link');
    const generateAIzipperBtn = document.getElementById('generate-aizipper-zip-btn'); // New button

    // Options controls
    const includeGACheck = document.getElementById('include-ga');
    const gaIdInput = document.getElementById('ga-id-input');
    const includeAdsenseCheck = document.getElementById('include-adsense');
    const adsClientInput = document.getElementById('ads-client-input');
    const includeCustomCheck = document.getElementById('include-custom');
    const customSnippetInput = document.getElementById('custom-snippet');
    const optionsSection = document.getElementById('options-section');
    const optionsHint = document.querySelector('.options-hint');

    // Drag and drop functionality
    function setupDragAndDrop(textareaElement) {
        textareaElement.addEventListener('dragover', (event) => {
            event.stopPropagation();
            event.preventDefault();
            event.dataTransfer.dropEffect = 'copy'; // Explicitly show this is a copy.
            textareaElement.classList.add('dragover'); // Optional: add a class for visual feedback
        });

        textareaElement.addEventListener('dragleave', (event) => {
            event.stopPropagation();
            event.preventDefault();
            textareaElement.classList.remove('dragover'); // Optional: remove visual feedback class
        });

        textareaElement.addEventListener('drop', (event) => {
            event.stopPropagation();
            event.preventDefault();
            textareaElement.classList.remove('dragover'); // Optional: remove visual feedback class

            const files = event.dataTransfer.files;
            if (files.length > 0) {
                const file = files[0];
                // Check if the file is a plain text file (or common code file types)
                // You might want to expand this list or make it more robust
                if (file.type.match('text.*') || file.name.endsWith('.html') || file.name.endsWith('.css') || file.name.endsWith('.js') || file.name.endsWith('.txt')) {
                    const reader = new FileReader();
                    reader.onload = (e) => {
                        textareaElement.value = e.target.result;
                    };
                    reader.readAsText(file);
                } else {
                    alert('Please drop a text file (e.g., .html, .css, .js, .txt).');
                }
            }
        });
    }

    setupDragAndDrop(htmlInput);
    setupDragAndDrop(cssInput);
    setupDragAndDrop(jsInput);

    // Toggle Options panel with ESC
    document.addEventListener('keydown', (event) => {
        if (event.key === 'Escape') {
            if (optionsSection) {
                const wasHidden = optionsSection.classList.contains('hidden');
                optionsSection.classList.toggle('hidden');
                if (wasHidden) {
                    // Auto-select all three checkboxes when opening, to speed up workflow
                    if (includeGACheck) includeGACheck.checked = true;
                    if (includeAdsenseCheck) includeAdsenseCheck.checked = true;
                    if (includeCustomCheck) includeCustomCheck.checked = true;
                    try { optionsSection.scrollIntoView({ behavior: 'smooth', block: 'start' }); } catch (e) {}
                }
            }
        }
    });

    // Helper: sanitize JS before injecting inline <script>
    function sanitizeInlineJs(js) {
        if (!js) return '';
        // If user mistakenly pasted HTML into JS textarea, skip injection to avoid syntax errors
        const looksLikeHtml = /<!DOCTYPE|<html[\s>]|<head[\s>]|<body[\s>]/i.test(js);
        if (looksLikeHtml) return null;
        // Prevent premature </script> termination
        return js.replace(/<\/script>/gi, '<\\/script>');
    }

    // --- Optional inclusions (GA, AdSense, custom snippet) helpers ---
    function injectIntoHead(html, snippet) {
        if (!snippet) return html;
        if (html.includes('</head>')) {
            return html.replace('</head>', `${snippet}\n</head>`);
        }
        // If no </head>, prepend as a fallback
        return `${snippet}\n` + html;
    }

    function injectIntoFooter(html, snippet) {
        if (!snippet) return html;
        if (html.includes('</body>')) {
            return html.replace('</body>', `${snippet}\n</body>`);
        }
        // If no </body>, append as a fallback
        return html + `\n${snippet}`;
    }

    function escapeHtml(text) {
        return String(text)
            .replace(/&/g, '&')
            .replace(/</g, '<')
            .replace(/>/g, '>')
            .replace(/"/g, '"')
            .replace(/'/g, '&#39;');
    }

    function linkifyPlainText(text) {
        if (!text) return '';
        const urlRegex = /\b(?:https?:\/\/|www\.)[^\s<>"']+/gi;
        let result = '';
        let lastIndex = 0;
        let match;
        while ((match = urlRegex.exec(text)) !== null) {
            const before = text.slice(lastIndex, match.index);
            result += escapeHtml(before);
            let url = match[0];
            const href = url.startsWith('www.') ? `https://${url}` : url;
            const safeHref = escapeHtml(href);
            const safeText = escapeHtml(url);
            result += `<a href="${safeHref}" target="_blank" rel="noopener noreferrer">${safeText}</a>`;
            lastIndex = urlRegex.lastIndex;
        }
        result += escapeHtml(text.slice(lastIndex));
        return result;
    }

    function buildCustomFooterHtml(raw) {
        const value = (raw || '').trim();
        if (!value) return '';
        let html;
        if (/[<][a-z!/]/i.test(value)) {
            // Looks like HTML; inject as-is
            html = value;
        } else {
            // Plain text; escape and linkify URLs
            html = `<p>${linkifyPlainText(value)}</p>`;
        }
        return `<!-- AIzipper Custom Footer Start -->
<div class="aizipper-custom-footer">
${html}
</div>
<!-- AIzipper Custom Footer End -->`;
    }

    function buildGASnippet(measurementId) {
        const id = (measurementId || '').trim();
        if (!id) return '';
        return `
<script async="true" src="https://www.googletagmanager.com/gtag/js?id=${id}"></script>
<script>
  window.dataLayer = window.dataLayer || [];
  function gtag(){dataLayer.push(arguments);}
  gtag('js', new Date());

  gtag('config', '${id}');
</script>`.trim();
    }

    function buildAdSenseSnippet(clientId) {
        const cid = (clientId || '').trim();
        if (!cid) return '';
        return `
<script data-ad-client="${cid}" async="true" src="https://pagead2.googlesyndication.com/pagead/js/adsbygoogle.js"></script>`.trim();
    }

    function applyOptionalInclusions(html) {
        try {
            let out = html;

            // Skip if content already contains GA/AdSense to avoid duplicates
            const hasGA = /googletagmanager\.com\/gtag\/js\?id=/i.test(out) || /gtag\(\s*['"]config['"]\s*,\s*['"][^'"]+['"]\s*\)/i.test(out);
            const hasAdsense = /pagead2\.googlesyndication\.com\/pagead\/js\/adsbygoogle\.js/i.test(out) || /data-ad-client\s*=\s*['"][^'"]+['"]/i.test(out);

            if (includeGACheck && includeGACheck.checked && !hasGA) {
                const gaId = (gaIdInput && gaIdInput.value || '').trim();
                const gaSnippet = buildGASnippet(gaId);
                if (gaSnippet) out = injectIntoHead(out, gaSnippet);
            }

            if (includeAdsenseCheck && includeAdsenseCheck.checked && !hasAdsense) {
                const clientId = (adsClientInput && adsClientInput.value || '').trim();
                const adSnippet = buildAdSenseSnippet(clientId);
                if (adSnippet) out = injectIntoHead(out, adSnippet);
            }

            const hasCustomFooter = /class=["']aizipper-custom-footer["']/i.test(out);
            if (includeCustomCheck && includeCustomCheck.checked && !hasCustomFooter) {
                const raw = (customSnippetInput && customSnippetInput.value || '').trim();
                const customSnippet = buildCustomFooterHtml(raw);
                if (customSnippet) out = injectIntoFooter(out, customSnippet);
            }

            return out;
        } catch (e) {
            console.warn('applyOptionalInclusions failed, returning original HTML.', e);
            return html;
        }
    }
    
    // Tab switching functionality
    tabButtons.forEach(button => {
        button.addEventListener('click', () => {
            // Remove active class from all buttons and sections
            tabButtons.forEach(btn => btn.classList.remove('active'));
            inputSections.forEach(section => section.classList.remove('active'));
            
            // Add active class to clicked button and corresponding section
            const tabName = button.getAttribute('data-tab');
            button.classList.add('active');
            document.getElementById(`${tabName}-section`).classList.add('active');
        });
    });

    // Preview HTML when preview button is clicked
    previewBtn.addEventListener('click', () => {
        const htmlCode = htmlInput.value; // Keep original for textarea
        const cssCode = cssInput.value.trim();
        const jsCode = jsInput.value.trim();

        if (!htmlCode.trim()) {
            alert('Please paste some HTML code first!');
            return;
        }

        previewContainer.innerHTML = '';
        const iframe = document.createElement('iframe');
        iframe.style.width = '100%';
        // iframe.style.height = '400px'; // Removed fixed height
        iframe.style.border = '1px solid #ddd';
        previewContainer.appendChild(iframe);

        const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
        try {
            iframeDoc.open();
        } catch (e) {
            // Fallback if some browsers object to open after append
        }

        let contentForPreview = htmlCode.trim();

        // Embed CSS if provided
        if (cssCode !== '') {
            if (contentForPreview.includes('</head>')) {
                contentForPreview = contentForPreview.replace('</head>', `<style>\n${cssCode}\n</style>\n</head>`);
            } else {
                contentForPreview = `<style>\n${cssCode}\n</style>\n` + contentForPreview;
            }
        }

        // Embed JS if provided (with sanitation)
        if (jsCode !== '') {
            const safeJs = sanitizeInlineJs(jsCode);
            if (safeJs === null) {
                console.warn('JS textarea appears to contain HTML. Skipping JS injection to prevent syntax errors.');
            } else {
                if (contentForPreview.includes('</body>')) {
                    contentForPreview = contentForPreview.replace('</body>', `<script>\n${safeJs}\n</script>\n</body>`);
                } else {
                    contentForPreview += `\n<script>\n${safeJs}\n</script>`;
                }
            }
        }
        
        // Apply optional inclusions (GA, AdSense, custom snippet)
        contentForPreview = applyOptionalInclusions(contentForPreview);
        
        try {
            // Prefer srcdoc when available for cleaner injection
            if ('srcdoc' in iframe) {
                iframe.srcdoc = contentForPreview;
            } else {
                iframeDoc.write(contentForPreview);
                iframeDoc.close();
            }
        } catch (e) {
            console.warn('Iframe write failed, falling back to srcdoc if supported.', e);
            if ('srcdoc' in iframe) {
                iframe.srcdoc = contentForPreview;
            }
        }

        // Adjust iframe height to content
        iframe.onload = () => {
            try {
                const body = iframe.contentWindow.document.body;
                const html = iframe.contentWindow.document.documentElement;
                const height = Math.max( body.scrollHeight, body.offsetHeight, 
                                       html.clientHeight, html.scrollHeight, html.offsetHeight );
                iframe.style.height = height + 'px';
            } catch (e) {
                console.warn("Could not resize iframe to content, likely due to cross-origin restrictions if previewing external content directly.", e);
                iframe.style.height = '400px'; // Fallback height
            }
        };
        // For browsers that might not fire onload for srcdoc/write, try to set height after write
        // This is a bit of a race condition, onload is preferred.
        try {
            const body = iframe.contentWindow.document.body;
            const html = iframe.contentWindow.document.documentElement;
            // Timeout to allow rendering
            setTimeout(() => {
                 const height = Math.max( body.scrollHeight, body.offsetHeight, 
                                       html.clientHeight, html.scrollHeight, html.offsetHeight );
                iframe.style.height = height + 'px';
            }, 100); // Adjust timeout if needed
        } catch (e) {
            // Fallback if immediate access fails (e.g. about:blank not ready)
             if (!iframe.onload) { // if onload didn't get set or fire
                iframe.style.height = '400px'; // Fallback height
             }
        }


        previewSection.classList.remove('hidden');
        previewSection.scrollIntoView({ behavior: 'smooth' });
    });

    // Preview in new tab when "Preview in New Tab" button is clicked
    previewNewTabBtn.addEventListener('click', () => {
        const htmlCode = htmlInput.value;
        const cssCode = cssInput.value.trim();
        const jsCode = jsInput.value.trim();

        if (!htmlCode.trim()) {
            alert('Please paste some HTML code first!');
            return;
        }

        let contentForPreview = htmlCode.trim();

        // Embed CSS if provided
        if (cssCode !== '') {
            if (contentForPreview.includes('</head>')) {
                contentForPreview = contentForPreview.replace('</head>', `<style>\n${cssCode}\n</style>\n</head>`);
            } else {
                contentForPreview = `<style>\n${cssCode}\n</style>\n` + contentForPreview;
            }
        }

        // Embed JS if provided (with sanitation)
        if (jsCode !== '') {
            const safeJs = sanitizeInlineJs(jsCode);
            if (safeJs === null) {
                console.warn('JS textarea appears to contain HTML. Skipping JS injection to prevent syntax errors.');
            } else {
                if (contentForPreview.includes('</body>')) {
                    contentForPreview = contentForPreview.replace('</body>', `<script>\n${safeJs}\n</script>\n</body>`);
                } else {
                    contentForPreview += `\n<script>\n${safeJs}\n</script>`;
                }
            }
        }

        // Apply optional inclusions (GA, AdSense, custom snippet)
        contentForPreview = applyOptionalInclusions(contentForPreview);

        const blob = new Blob([contentForPreview], { type: 'text/html' });
        const url = URL.createObjectURL(blob);
        window.open(url, '_blank');
    });

    // Generate ZIP file when button is clicked
    generateBtn.addEventListener('click', () => {
        const htmlCode = htmlInput.value; // Keep original for textarea
        const cssCode = cssInput.value.trim();
        const jsCode = jsInput.value.trim();

        if (!htmlCode.trim()) {
            alert('Please paste some HTML code first!');
            return;
        }
        
        generateZip(htmlCode.trim(), cssCode, jsCode);
    });
    
    // Clear the input field
    clearBtn.addEventListener('click', () => {
        htmlInput.value = '';
        cssInput.value = '';
        jsInput.value = '';
        previewSection.classList.add('hidden');
        resultSection.classList.add('hidden');
    });

    // Generate AIzipper_YYYYMMDD.zip when new button is clicked
    if (generateAIzipperBtn) {
        generateAIzipperBtn.addEventListener('click', () => {
            const htmlCode = htmlInput.value;
            const cssCode = cssInput.value.trim();
            const jsCode = jsInput.value.trim();

            if (!htmlCode.trim()) {
                alert('Please provide HTML code in the HTML textarea to generate the ZIP.');
                return;
            }
            generateAIzipperFile(htmlCode.trim(), cssCode, jsCode);
        });
    }
    
    // Function to generate ZIP file
    async function generateZip(htmlCode, cssCode, jsCode) {
        try {
            const zip = new JSZip();
            let updatedHtmlCode = htmlCode;

            // Add styles.css if CSS code is provided and update HTML links
            if (cssCode !== '') {
                zip.file('styles.css', cssCode);
                // Regex to find and replace CSS links. Handles various quote types and spacing.
                // It looks for <link ... href="some-style.css" ...> or <link ... href='some-style.css' ...>
                updatedHtmlCode = updatedHtmlCode.replace(/<link\s+[^>]*href\s*=\s*['"]([^'"]*\.(?:css|scss|less))['"][^>]*>/gi, (match, oldHref) => {
                    const lower = oldHref.toLowerCase().trim();
                    const isExternal = /^(https?:)?\/\//.test(lower) || lower.startsWith('data:');
                    // Only replace relative paths, leave CDNs/data URIs untouched
                    if (!isExternal && lower !== 'styles.css') {
                        return match.replace(oldHref, 'styles.css');
                    }
                    return match; // Return original match if it's already 'styles.css' or external
                });
                // If no link tag for 'styles.css' exists after replacement and no inline <style> tag, add one.
                if (!updatedHtmlCode.match(/<link\s+[^>]*href\s*=\s*['"]styles\.css['"][^>]*>/gi) && !updatedHtmlCode.includes('<style>')) {
                    if (updatedHtmlCode.includes('</head>')) {
                        updatedHtmlCode = updatedHtmlCode.replace('</head>', '<link rel="stylesheet" href="styles.css">\n</head>');
                    } else {
                        // Fallback if no </head> tag, prepend to the document (less ideal)
                        updatedHtmlCode = '<link rel="stylesheet" href="styles.css">\n' + updatedHtmlCode;
                    }
                }
            }
            
            // Add script.js if JavaScript code is provided and update HTML links
            if (jsCode !== '') {
                zip.file('script.js', jsCode);
                // Regex to find and replace JS script src. Handles various quote types and spacing.
                // It looks for <script ... src="some-script.js" ...></script> or <script ... src='some-script.js' ...></script>
                updatedHtmlCode = updatedHtmlCode.replace(/<script\s+[^>]*src\s*=\s*['"]([^'"]*\.js)['"][^>]*>\s*<\/script>/gi, (match, oldSrc) => {
                    const lower = oldSrc.toLowerCase().trim();
                    const isExternal = /^(https?:)?\/\//.test(lower) || lower.startsWith('data:');
                    // Only replace relative paths, leave CDNs/data URIs untouched
                    if (!isExternal && lower !== 'script.js') {
                        return match.replace(oldSrc, 'script.js');
                    }
                    return match; // Return original match if it's already 'script.js' or external
                });
                // If no script tag for 'script.js' exists after replacement and no inline <script> (without src), add one.
                if (!updatedHtmlCode.match(/<script\s+[^>]*src\s*=\s*['"]script\.js['"][^>]*>\s*<\/script>/gi) && !updatedHtmlCode.match(/<script\s*(?!src)[^>]*>[\s\S]*?<\/script>/gi)) {
                    if (updatedHtmlCode.includes('</body>')) {
                        updatedHtmlCode = updatedHtmlCode.replace('</body>', '<script src="script.js"></script>\n</body>');
                    } else {
                        // Fallback if no </body> tag, append to the document (less ideal)
                        updatedHtmlCode += '\n<script src="script.js"></script>';
                    }
                }
            }
 
            // Apply optional inclusions (GA, AdSense, custom snippet)
            updatedHtmlCode = applyOptionalInclusions(updatedHtmlCode);

            // Always add index.html (potentially updated)
            zip.file('index.html', updatedHtmlCode);
            
            // Generate the ZIP file
            const zipBlob = await zip.generateAsync({ type: 'blob' });
            
            // Extract title from HTML content
            let title = 'gemini_code'; // Default title
            const titleMatch = updatedHtmlCode.match(/<title>([^<]+)<\/title>/i);
            if (titleMatch && titleMatch[1]) {
                title = titleMatch[1].trim().replace(/\s+/g, '_'); // Replace spaces with underscores
            }
            
            // Create a timestamp for the filename (YYYYMMDD format)
            const now = new Date();
            const year = now.getFullYear();
            const month = String(now.getMonth() + 1).padStart(2, '0');
            const day = String(now.getDate()).padStart(2, '0');
            const timestamp = `${year}${month}${day}`;
            
            // Create a meaningful filename with title and timestamp
            const filename = `${title}_${timestamp}.zip`;
            
            // Create a download URL for the ZIP file
            const downloadUrl = URL.createObjectURL(zipBlob);
            
            // Update the UI
            filenameDisplay.textContent = filename;
            downloadLink.href = downloadUrl;
            downloadLink.download = filename;
            resultSection.classList.remove('hidden');
            
            // Scroll to the result section
            resultSection.scrollIntoView({ behavior: 'smooth' });
        } catch (error) {
            console.error('Error generating ZIP file:', error);
            alert('An error occurred while generating the ZIP file. Please try again.');
        }
    }

    // Function to generate AIzipper_YYYYMMDD.zip
    async function generateAIzipperFile(htmlCode, cssCode, jsCode) {
        try {
            const zip = new JSZip();
            
            // Apply optional inclusions (GA, AdSense, custom snippet)
            const finalHtml = applyOptionalInclusions(htmlCode);

            zip.file('index.html', finalHtml); 
            
            if (cssCode.trim() !== '') {
                zip.file('styles.css', cssCode.trim());
            }
            
            if (jsCode.trim() !== '') {
                zip.file('script.js', jsCode.trim());
            }
            
            const zipBlob = await zip.generateAsync({ type: 'blob' });
            
            const now = new Date();
            const year = now.getFullYear();
            const month = String(now.getMonth() + 1).padStart(2, '0');
            const day = String(now.getDate()).padStart(2, '0');
            const timestamp = `${year}${month}${day}`;
            
            const filename = `AIzipper_${timestamp}.zip`;
            
            const downloadUrl = URL.createObjectURL(zipBlob);
            const tempLink = document.createElement('a');
            tempLink.href = downloadUrl;
            tempLink.download = filename;
            document.body.appendChild(tempLink);
            tempLink.click();
            document.body.removeChild(tempLink);
            URL.revokeObjectURL(downloadUrl);

            resultSection.classList.add('hidden');
            alert(`Downloading ${filename}...`); 
        } catch (error) {
            console.error('Error generating AIzipper ZIP file:', error);
            alert('An error occurred while generating the AIzipper ZIP file. Please try again.');
        }
    }
});
