document.addEventListener('DOMContentLoaded', () => {
    // Get DOM elements
    const htmlInput = document.getElementById('html-input');
    const cssInput = document.getElementById('css-input');
    const jsInput = document.getElementById('js-input');
    // const fileStructure = document.getElementById('file-structure'); // Removed
    const tabButtons = document.querySelectorAll('.tab-btn');
    const inputSections = document.querySelectorAll('.input-section');
    const previewBtn = document.getElementById('preview-btn');
    const generateBtn = document.getElementById('generate-zip');
    const clearBtn = document.getElementById('clear-btn');
    const previewSection = document.getElementById('preview-section');
    const previewContainer = document.getElementById('preview-container');
    const resultSection = document.getElementById('result-section');
    const filenameDisplay = document.getElementById('filename-display');
    const downloadLink = document.getElementById('download-link');
    
    // Tab switching functionality
    tabButtons.forEach(button => {
        button.addEventListener('click', () => {
            // Remove active class from all buttons and sections
            tabButtons.forEach(btn => btn.classList.remove('active'));
            inputSections.forEach(section => section.classList.remove('active'));
            
            // Add active class to clicked button and corresponding section
            const tabName = button.getAttribute('data-tab');
            button.classList.add('active');
            document.getElementById(`${tabName}-section`).classList.add('active');
        });
    });

    // Preview HTML when preview button is clicked
    previewBtn.addEventListener('click', () => {
        const htmlCode = htmlInput.value; // Keep original for textarea
        const cssCode = cssInput.value.trim();
        const jsCode = jsInput.value.trim();

        if (!htmlCode.trim()) {
            alert('Please paste some HTML code first!');
            return;
        }

        previewContainer.innerHTML = '';
        const iframe = document.createElement('iframe');
        iframe.style.width = '100%';
        // iframe.style.height = '400px'; // Removed fixed height
        iframe.style.border = '1px solid #ddd';
        previewContainer.appendChild(iframe);

        const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
        iframeDoc.open();

        let contentForPreview = htmlCode.trim();

        // Embed CSS if provided
        if (cssCode !== '') {
            if (contentForPreview.includes('</head>')) {
                contentForPreview = contentForPreview.replace('</head>', `<style>\n${cssCode}\n</style>\n</head>`);
            } else {
                contentForPreview = `<style>\n${cssCode}\n</style>\n` + contentForPreview;
            }
        }

        // Embed JS if provided
        if (jsCode !== '') {
            if (contentForPreview.includes('</body>')) {
                contentForPreview = contentForPreview.replace('</body>', `<script>\n${jsCode}\n</script>\n</body>`);
            } else {
                contentForPreview += `\n<script>\n${jsCode}\n</script>`;
            }
        }
        
        iframeDoc.write(contentForPreview);
        iframeDoc.close();

        // Adjust iframe height to content
        iframe.onload = () => {
            try {
                const body = iframe.contentWindow.document.body;
                const html = iframe.contentWindow.document.documentElement;
                const height = Math.max( body.scrollHeight, body.offsetHeight, 
                                       html.clientHeight, html.scrollHeight, html.offsetHeight );
                iframe.style.height = height + 'px';
            } catch (e) {
                console.warn("Could not resize iframe to content, likely due to cross-origin restrictions if previewing external content directly.", e);
                iframe.style.height = '400px'; // Fallback height
            }
        };
        // For browsers that might not fire onload for srcdoc/write, try to set height after write
        // This is a bit of a race condition, onload is preferred.
        try {
            const body = iframe.contentWindow.document.body;
            const html = iframe.contentWindow.document.documentElement;
            // Timeout to allow rendering
            setTimeout(() => {
                 const height = Math.max( body.scrollHeight, body.offsetHeight, 
                                       html.clientHeight, html.scrollHeight, html.offsetHeight );
                iframe.style.height = height + 'px';
            }, 100); // Adjust timeout if needed
        } catch (e) {
            // Fallback if immediate access fails (e.g. about:blank not ready)
             if (!iframe.onload) { // if onload didn't get set or fire
                iframe.style.height = '400px'; // Fallback height
             }
        }


        previewSection.classList.remove('hidden');
        previewSection.scrollIntoView({ behavior: 'smooth' });
    });

    // Generate ZIP file when button is clicked
    generateBtn.addEventListener('click', () => {
        const htmlCode = htmlInput.value; // Keep original for textarea
        const cssCode = cssInput.value.trim();
        const jsCode = jsInput.value.trim();

        if (!htmlCode.trim()) {
            alert('Please paste some HTML code first!');
            return;
        }
        
        generateZip(htmlCode.trim(), cssCode, jsCode);
    });
    
    // Clear the input field
    clearBtn.addEventListener('click', () => {
        htmlInput.value = '';
        cssInput.value = '';
        jsInput.value = '';
        previewSection.classList.add('hidden');
        resultSection.classList.add('hidden');
    });
    
    // Function to generate ZIP file
    async function generateZip(htmlCode, cssCode, jsCode) {
        try {
            const zip = new JSZip();
            
            // Always add index.html
            zip.file('index.html', htmlCode);
            
            // Add styles.css if CSS code is provided
            if (cssCode.trim() !== '') {
                zip.file('styles.css', cssCode.trim());
            }
            
            // Add script.js if JavaScript code is provided
            if (jsCode.trim() !== '') {
                zip.file('script.js', jsCode.trim());
            }
            
            const zipBlob = await zip.generateAsync({ type: 'blob' });
            
            let title = 'gemini_code';
            const titleMatch = htmlCode.match(/<title>([^<]+)<\/title>/i);
            if (titleMatch && titleMatch[1]) {
                title = titleMatch[1].trim().replace(/\s+/g, '_').replace(/[^\w-]/g, ''); // Sanitize title
                if (!title) title = 'untitled'; // Ensure title is not empty after sanitization
            }
            
            const now = new Date();
            const year = now.getFullYear();
            const month = String(now.getMonth() + 1).padStart(2, '0');
            const day = String(now.getDate()).padStart(2, '0');
            const timestamp = `${year}${month}${day}`;
            
            const filename = `${title}_${timestamp}.zip`;
            
            const downloadUrl = URL.createObjectURL(zipBlob);
            
            // Create a temporary link to trigger the download
            const tempLink = document.createElement('a');
            tempLink.href = downloadUrl;
            tempLink.download = filename;
            document.body.appendChild(tempLink);
            tempLink.click();
            document.body.removeChild(tempLink);
            URL.revokeObjectURL(downloadUrl); // Clean up the object URL

            // Hide the result section as the download starts immediately
            resultSection.classList.add('hidden');
            // No need to scroll, as there's nothing new to show.
            // An alert or a subtle notification could be added here if desired.
            // For now, per request, it just downloads.
            alert(`Downloading ${filename}...`); // Optional: provide some feedback
        } catch (error) {
            console.error('Error generating ZIP file:', error);
            alert('An error occurred while generating the ZIP file. Please try again.');
        }
    }
});