// Game data - vocabulary words and their meanings organized by levels
const gameLevels = [
    // Level 1 - Basic vocabulary
    [
        { word: "Happy", meaning: "Feeling joy or pleasure" },
        { word: "Sad", meaning: "Feeling unhappy or sorrowful" },
        { word: "Big", meaning: "Large in size" },
        { word: "Small", meaning: "Little in size" },
        { word: "Fast", meaning: "Moving quickly" }
    ],
    // Level 2 - Intermediate vocabulary
    [
        { word: "Brave", meaning: "Showing courage" },
        { word: "Clever", meaning: "Quick to understand" },
        { word: "Gentle", meaning: "Kind and soft" },
        { word: "Shiny", meaning: "Reflecting light" },
        { word: "Noisy", meaning: "Making loud sounds" },
        { word: "Quiet", meaning: "Making little sound" }
    ],
    // Level 3 - Advanced vocabulary
    [
        { word: "Curious", meaning: "Eager to learn" },
        { word: "Generous", meaning: "Willing to give" },
        { word: "Mysterious", meaning: "Difficult to understand" },
        { word: "Magnificent", meaning: "Extremely beautiful" },
        { word: "Enormous", meaning: "Extremely large" },
        { word: "Delicious", meaning: "Very tasty" },
        { word: "Ancient", meaning: "Very old" }
    ]
];

// Game state variables
let currentLevel = 0;
let score = 0;
let selectedWord = null;
let selectedMeaning = null;
let correctMatches = [];
let attempts = 0;
let starsEarned = 0;

// DOM elements
const wordContainer = document.getElementById('word-container');
const meaningContainer = document.getElementById('meaning-container');
const scoreElement = document.getElementById('score');
const levelElement = document.getElementById('level');
const feedbackMessage = document.getElementById('feedback-message');
const checkButton = document.getElementById('check-button');
const nextLevelButton = document.getElementById('next-level-button');
const resetButton = document.getElementById('reset-button');
const stars = [
    document.getElementById('star1'),
    document.getElementById('star2'),
    document.getElementById('star3')
];

// Initialize the game
function initGame() {
    // Reset game state
    currentLevel = 0;
    score = 0;
    correctMatches = [];
    attempts = 0;
    starsEarned = 0;
    
    // Update UI
    updateScore();
    updateLevel();
    resetStars();
    
    // Load the first level
    loadLevel(currentLevel);
    
    // Reset button states
    checkButton.disabled = false;
    nextLevelButton.disabled = true;
}

// Load a specific level
function loadLevel(level) {
    // Clear previous level content
    wordContainer.innerHTML = '';
    meaningContainer.innerHTML = '';
    feedbackMessage.textContent = '';
    correctMatches = [];
    attempts = 0;
    
    // Get the vocabulary pairs for this level
    const vocabularyPairs = gameLevels[level];
    
    // Create and shuffle arrays of words and meanings
    const words = vocabularyPairs.map(pair => pair.word);
    const meanings = vocabularyPairs.map(pair => pair.meaning);
    
    // Shuffle the arrays
    shuffleArray(words);
    shuffleArray(meanings);
    
    // Create word cards
    words.forEach(word => {
        const wordCard = document.createElement('div');
        wordCard.className = 'word-card';
        wordCard.textContent = word;
        wordCard.dataset.word = word;
        wordCard.addEventListener('click', () => selectWord(wordCard));
        wordContainer.appendChild(wordCard);
    });
    
    // Create meaning cards
    meanings.forEach(meaning => {
        const meaningCard = document.createElement('div');
        meaningCard.className = 'meaning-card';
        meaningCard.textContent = meaning;
        meaningCard.dataset.meaning = meaning;
        meaningCard.addEventListener('click', () => selectMeaning(meaningCard));
        meaningContainer.appendChild(meaningCard);
    });
}

// Handle word card selection
function selectWord(wordCard) {
    // Deselect previously selected word if any
    const previouslySelectedWord = document.querySelector('.word-card.selected');
    if (previouslySelectedWord) {
        previouslySelectedWord.classList.remove('selected');
    }
    
    // Select the clicked word
    wordCard.classList.add('selected');
    selectedWord = wordCard.dataset.word;
    
    // Check if we have both a word and meaning selected
    checkSelection();
}

// Handle meaning card selection
function selectMeaning(meaningCard) {
    // Deselect previously selected meaning if any
    const previouslySelectedMeaning = document.querySelector('.meaning-card.selected');
    if (previouslySelectedMeaning) {
        previouslySelectedMeaning.classList.remove('selected');
    }
    
    // Select the clicked meaning
    meaningCard.classList.add('selected');
    selectedMeaning = meaningCard.dataset.meaning;
    
    // Check if we have both a word and meaning selected
    checkSelection();
}

// Check if both a word and meaning are selected
function checkSelection() {
    if (selectedWord && selectedMeaning) {
        // Enable the check button if both are selected
        checkButton.disabled = false;
    }
}

// Check if the selected word and meaning match
function checkMatch() {
    attempts++;
    
    // Find the correct meaning for the selected word
    const vocabularyPairs = gameLevels[currentLevel];
    const correctPair = vocabularyPairs.find(pair => pair.word === selectedWord);
    
    if (correctPair && correctPair.meaning === selectedMeaning) {
        // Correct match
        handleCorrectMatch();
    } else {
        // Incorrect match
        handleIncorrectMatch();
    }
    
    // Reset selections
    selectedWord = null;
    selectedMeaning = null;
    
    // Check if level is complete
    checkLevelCompletion();
}

// Handle correct match
function handleCorrectMatch() {
    // Find the selected elements
    const wordCard = document.querySelector(`.word-card[data-word="${selectedWord}"]`);
    const meaningCard = document.querySelector(`.meaning-card[data-meaning="${selectedMeaning}"]`);
    
    // Add to correct matches
    correctMatches.push(selectedWord);
    
    // Update UI for correct match
    wordCard.classList.remove('selected');
    meaningCard.classList.remove('selected');
    wordCard.classList.add('correct');
    meaningCard.classList.add('correct');
    
    // Disable the matched cards
    wordCard.style.pointerEvents = 'none';
    meaningCard.style.pointerEvents = 'none';
    
    // Update score
    score += 1;
    if (score > 99) score = 99; // Cap score at 99
    updateScore();
    
    // Show feedback
    feedbackMessage.textContent = 'Correct! Great job! 🎉';
    feedbackMessage.style.color = '#22c55e';
    
    // Play success sound effect
    playSound('success');
}

// Handle incorrect match
function handleIncorrectMatch() {
    // Find the selected elements
    const wordCard = document.querySelector(`.word-card[data-word="${selectedWord}"]`);
    const meaningCard = document.querySelector(`.meaning-card[data-meaning="${selectedMeaning}"]`);
    
    // Update UI for incorrect match
    wordCard.classList.remove('selected');
    meaningCard.classList.remove('selected');
    
    // Temporarily show incorrect feedback
    wordCard.classList.add('incorrect');
    meaningCard.classList.add('incorrect');
    
    // Remove incorrect styling after animation
    setTimeout(() => {
        wordCard.classList.remove('incorrect');
        meaningCard.classList.remove('incorrect');
    }, 1000);
    
    // Show feedback
    feedbackMessage.textContent = 'Not quite right. Try again! 💪';
    feedbackMessage.style.color = '#ef4444';
    
    // Play error sound effect
    playSound('error');
}

// Check if the current level is complete
function checkLevelCompletion() {
    const vocabularyPairs = gameLevels[currentLevel];
    
    if (correctMatches.length === vocabularyPairs.length) {
        // Level complete
        feedbackMessage.textContent = 'Level Complete! 🌟';
        feedbackMessage.style.color = '#3b82f6';
        
        // Award stars based on performance
        awardStars();
        
        // Enable next level button if not on the last level
        if (currentLevel < gameLevels.length - 1) {
            nextLevelButton.disabled = false;
        } else {
            feedbackMessage.textContent = 'Congratulations! You completed all levels! 🏆';
            playSound('victory');
        }
        
        // Disable check button
        checkButton.disabled = true;
    }
}

// Award stars based on performance
function awardStars() {
    const vocabularyPairs = gameLevels[currentLevel];
    const perfectScore = vocabularyPairs.length;
    
    // Calculate stars based on attempts
    // 3 stars: Perfect (no mistakes)
    // 2 stars: Good (few mistakes)
    // 1 star: Completed (many mistakes)
    let newStars = 0;
    
    if (attempts === perfectScore) {
        newStars = 3; // Perfect score
    } else if (attempts <= perfectScore * 1.5) {
        newStars = 2; // Good score
    } else {
        newStars = 1; // Completed
    }
    
    // Update stars display
    for (let i = 0; i < newStars; i++) {
        setTimeout(() => {
            stars[i].classList.add('earned');
            playSound('star');
        }, i * 500);
    }
    
    starsEarned = newStars;
}

// Move to the next level
function nextLevel() {
    currentLevel++;
    updateLevel();
    resetStars();
    loadLevel(currentLevel);
    nextLevelButton.disabled = true;
    checkButton.disabled = false;
}

// Reset stars display
function resetStars() {
    stars.forEach(star => star.classList.remove('earned'));
}

// Update the score display
function updateScore() {
    scoreElement.textContent = score;
    
    // Animate score change
    scoreElement.classList.add('score-change');
    setTimeout(() => {
        scoreElement.classList.remove('score-change');
    }, 500);
}

// Update the level display
function updateLevel() {
    levelElement.textContent = currentLevel + 1;
}

// Utility function to shuffle an array
function shuffleArray(array) {
    for (let i = array.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1));
        [array[i], array[j]] = [array[j], array[i]];
    }
    return array;
}

// Play sound effects (placeholder function - would need actual audio files)
function playSound(type) {
    // This is a placeholder - in a real implementation, you would play actual sound files
    // For example:
    // const sound = new Audio(`sounds/${type}.mp3`);
    // sound.play();
    
    // For now, we'll just log to console
    console.log(`Playing ${type} sound`);
}

// Event listeners
checkButton.addEventListener('click', checkMatch);
nextLevelButton.addEventListener('click', nextLevel);
resetButton.addEventListener('click', initGame);

// Initialize the game when the page loads
document.addEventListener('DOMContentLoaded', initGame);