
import React, { useState, useCallback } from 'react';
import { SCENARIOS } from './constants';
import type { Scenario, Idiom } from './types';
import { getFeedbackForIdiom } from './services/geminiService';
import ScenarioView from './components/ScenarioView';
import FeedbackBubble from './components/FeedbackBubble';
import { Header } from './components/Header';
import { ScenarioSelector } from './components/ScenarioSelector';

const App: React.FC = () => {
  const [activeScenarioIndex, setActiveScenarioIndex] = useState<number>(0);
  const [selectedIdiom, setSelectedIdiom] = useState<Idiom | null>(null);
  const [feedback, setFeedback] = useState<string>('');
  const [isLoading, setIsLoading] = useState<boolean>(false);
  const [error, setError] = useState<string>('');

  const activeScenario: Scenario = SCENARIOS[activeScenarioIndex];

  const handleSelectScenario = (index: number) => {
    setActiveScenarioIndex(index);
    setSelectedIdiom(null);
    setFeedback('');
    setError('');
  };

  const handleIdiomSelect = useCallback(async (idiom: Idiom) => {
    if (isLoading) return;

    setSelectedIdiom(idiom);
    setIsLoading(true);
    setFeedback('');
    setError('');

    try {
      const { sentence, targetWord } = activeScenario;
      const responseText = await getFeedbackForIdiom(sentence, targetWord, idiom.word);
      setFeedback(responseText);
    } catch (err) {
      console.error("Error getting feedback:", err);
      setError('哎呀，老师好像有点忙，请稍后再试！');
    } finally {
      setIsLoading(false);
    }
  }, [activeScenario, isLoading]);

  return (
    <div className="min-h-screen bg-sky-50 text-gray-800 antialiased">
      <main className="container mx-auto px-4 py-8 max-w-4xl">
        <Header />
        
        <ScenarioSelector 
          scenarios={SCENARIOS}
          activeIndex={activeScenarioIndex}
          onSelect={handleSelectScenario}
        />

        <div className="mt-8">
          <ScenarioView 
            scenario={activeScenario}
            selectedIdiom={selectedIdiom}
            onIdiomSelect={handleIdiomSelect}
            isLoading={isLoading}
          />
        </div>

        <div className="mt-6 min-h-[100px]">
          <FeedbackBubble
            feedback={feedback}
            error={error}
            isLoading={isLoading}
          />
        </div>
      </main>
    </div>
  );
};

export default App;
