(function () {
  'use strict';

  // Safe helper that only applies URL-provided config if all required params exist.
  // This prevents errors when the page is opened without launch parameters.
  const XAPIUtils = {
    parameters: null,

    getParameters: function () {
      if (this.parameters) return this.parameters;

      try {
        const urlParams = new URLSearchParams(window.location.search);
        const endpoint = urlParams.get('endpoint');
        const auth = urlParams.get('auth');
        const agentRaw = urlParams.get('agent');
        const stateId = urlParams.get('stateId');
        const activityId = urlParams.get('activityId');

        // Only configure ADL.XAPIWrapper if endpoint and auth are present
        if (endpoint && auth && typeof window.ADL !== 'undefined' && window.ADL.XAPIWrapper) {
          const ep = endpoint.endsWith('/') ? endpoint : `${endpoint}/`;
          // Keep behavior consistent with original: auth is expected to be the base64 "user:pass"
          window.ADL.XAPIWrapper.changeConfig({
            endpoint: ep,
            auth: `Basic ${auth}`
          });
        }

        // Parse agent only if present and valid JSON
        let agent = null;
        if (agentRaw) {
          try {
            agent = JSON.parse(agentRaw);
          } catch (e) {
            console.warn('Invalid agent JSON in URL parameter. Ignoring.', e);
          }
        }

        this.parameters = { agent, stateId, activityId, endpoint, auth };
      } catch (e) {
        console.warn('XAPIUtils.getParameters() failed:', e);
        this.parameters = null;
      }

      return this.parameters;
    }
  };

  // Expose on window
  window.XAPIUtils = XAPIUtils;

  // Optionally initialize from URL params on load if endpoint+auth exist
  document.addEventListener('DOMContentLoaded', function () {
    const params = XAPIUtils.getParameters();
    if (!params || !params.endpoint || !params.auth) {
      // No launch params present; do nothing. The main app's UI will configure ADL.XAPIWrapper.
      return;
    }
  });

  // Optional helpers retained from original, made safe with guards.
  window.storeState = function storeState(stateValue) {
    try {
      if (!window.ADL || !window.ADL.XAPIWrapper) throw new Error('ADL.XAPIWrapper not available.');
      const parameters = XAPIUtils.getParameters() || {};
      const { activityId, stateId, agent } = parameters;
      if (!activityId || !stateId || !agent) throw new Error('Missing activityId, stateId, or agent.');

      const registration = null;
      window.ADL.XAPIWrapper.sendState(activityId, agent, stateId, registration, stateValue);
      console.log('Submitted:', stateValue);
    } catch (err) {
      console.error('storeState error:', err);
    }
  };

  window.getState = function getState() {
    try {
      if (!window.ADL || !window.ADL.XAPIWrapper) throw new Error('ADL.XAPIWrapper not available.');
      const parameters = XAPIUtils.getParameters() || {};
      const { activityId, stateId, agent } = parameters;
      if (!activityId || !stateId || !agent) throw new Error('Missing activityId, stateId, or agent.');

      const result = window.ADL.XAPIWrapper.getState(activityId, agent, stateId);
      const el = document.querySelector('#getState');
      if (el) el.innerText = 'First Load State: ' + JSON.stringify(result, null, 2);
      return result;
    } catch (err) {
      console.error('getState error:', err);
      const el = document.querySelector('#getState');
      if (el) el.innerText = 'Error has occurred: ' + err;
      return null;
    }
  };
})();
