'use strict';

// xAPI Integrator - Pure client-side (no server)
// - Reads an uploaded ZIP with JSZip
// - Finds the folder containing index.html
// - Injects xAPI libs and optional instrumentation snippet
// - Removes CSP-blocked GA/AdSense scripts
// - Adds vendored libs into lib/ within the content folder
// - Re-zips and offers a download link

(function () {
  const zipInput = document.getElementById('zipInput');
  const integrateBtn = document.getElementById('integrateBtn');
  const downloadLink = document.getElementById('downloadLink');
  const statusText = document.getElementById('statusText');
  const logEl = document.getElementById('log');

  function log(msg) {
    try { console.log('[integrator]', msg); } catch {}
    if (logEl) {
      logEl.textContent += (logEl.textContent ? '\n' : '') + msg;
    }
  }

  function setStatus(msg) {
    if (statusText) statusText.textContent = msg || '';
  }

  function getSelectedMode() {
    const el = document.querySelector('input[name="mode"]:checked');
    return (el && el.value) || 'timeline';
  }

  function joinPath(...parts) {
    return parts
      .filter(Boolean)
      .map(p => String(p).replace(/^\/+|\/+$/g, ''))
      .filter(p => p.length > 0)
      .join('/')
      + (String(parts[parts.length - 1]).endsWith('/') ? '/' : '');
  }

  // Load vendor libs from ./vendor so we can place them into the output ZIP
  async function loadVendors() {
    async function fetchText(url) {
      const res = await fetch(url);
      if (!res.ok) throw new Error(`Failed to fetch ${url}: ${res.status} ${res.statusText}`);
      return await res.text();
    }

    // In file:// contexts, fetch may be blocked. Advise to serve over http(s).
    try {
      const [wrapper, glue] = await Promise.all([
        fetchText('./vendor/xapiwrapper.min.js'),
        fetchText('./vendor/xAPI.js'),
      ]);
      return { wrapper, glue };
    } catch (e) {
      // Fallback: try vendor files from ../assets/lib relative to public/
      try {
        const [wrapper2, glue2] = await Promise.all([
          fetchText('../assets/lib/xapiwrapper.min.js'),
          fetchText('../assets/lib/xAPI.js'),
        ]);
        log('Loaded vendor scripts from ../assets/lib fallback.');
        return { wrapper: wrapper2, glue: glue2 };
      } catch (e2) {
        setStatus('Failed to load vendor scripts. If you opened this page via file://, please serve over HTTP(S) (e.g., npx http-server xapi-integrator/public) or deploy to any static host.');
        log('Error loading vendor scripts. If opening via file://, run a local web server (e.g., `npx http-server`).');
        throw e;
      }
    }
  }

  // Find content dir containing index.html (root, single subfolder, or shallow search)
  function findContentDirFromZip(zip) {
    const files = Object.keys(zip.files); // paths with forward slashes
    const indexCandidates = files.filter(k => k.toLowerCase().endsWith('index.html'));

    if (indexCandidates.length === 0) return { contentDir: '', indexPath: null };

    // Choose the shortest (shallowest) path
    const sorted = indexCandidates.sort((a, b) => {
      const da = a.split('/').length;
      const db = b.split('/').length;
      if (da !== db) return da - db;
      return a.length - b.length;
    });

    const indexPath = sorted[0];
    const dir = indexPath.slice(0, indexPath.length - 'index.html'.length);
    return { contentDir: dir, indexPath };
  }

  function serializeDocument(doc) {
    // Preserve a doctype if one exists; if not, prefix with <!doctype html>
    const doctype = '<!doctype html>';
    return doctype + '\n' + doc.documentElement.outerHTML;
  }

  function ensureHeadBody(doc) {
    if (!doc.head) {
      const head = doc.createElement('head');
      const html = doc.documentElement || doc.querySelector('html');
      if (html) html.insertBefore(head, html.firstChild);
    }
    if (!doc.body) {
      const body = doc.createElement('body');
      const html = doc.documentElement || doc.querySelector('html');
      if (html) html.appendChild(body);
    }
  }

  function hasScriptBySrc(doc, endsWithPath) {
    const scripts = Array.from(doc.querySelectorAll('script[src]'));
    return scripts.some(s => {
      const src = s.getAttribute('src') || '';
      return src.endsWith(endsWithPath) || src.includes(endsWithPath);
    });
  }

  function injectScriptsIntoHtml(html, options) {
    const { mode = 'timeline' } = options || {};
    const parser = new DOMParser();
    const doc = parser.parseFromString(html, 'text/html');
    ensureHeadBody(doc);

    // Remove CSP-blocked scripts
    Array.from(doc.querySelectorAll('script[src*="googletagmanager.com"],script[src*="googlesyndication.com"]'))
      .forEach(s => s.remove());

    // Inject xAPI libs if missing
    if (!hasScriptBySrc(doc, 'lib/xapiwrapper.min.js')) {
      const s1 = doc.createElement('script');
      s1.setAttribute('src', './lib/xapiwrapper.min.js');
      doc.head.appendChild(s1);
    }
    if (!hasScriptBySrc(doc, 'lib/xAPI.js')) {
      const s2 = doc.createElement('script');
      s2.setAttribute('src', './lib/xAPI.js');
      doc.head.appendChild(s2);
    }

    // Inject instrumentation snippet (timeline mode only)
    if (mode === 'timeline') {
      const already = Array.from(doc.querySelectorAll('script'))
        .some(s => ((s.textContent || '').includes('__xapiIntegrator')));
      if (!already) {
        const snippet = `
(function(){
  if (window.__xapiIntegrator) return;
  window.__xapiIntegrator = true;

  const sessionStart = Date.now();
  const actionLog = [];
  const xapi = { score: 0, interactionCount: 0, playStart: null, totalPlayMs: 0 };

  // Pending payload that always reflects the latest feedback/score (for auto-send hooks)
  window.__xapiPending = null;

  function tSeconds() { return Math.round((Date.now() - sessionStart)/1000); }
  function flushPlayTime() {
    if (xapi.playStart) { xapi.totalPlayMs += (Date.now() - xapi.playStart); xapi.playStart = null; }
  }
  function logAction(message) {
    const t = tSeconds();
    actionLog.push({ t, message });
    try { console.log('[xAPI Integrator]', 't=' + t + ',', message); } catch(e){}
    updateFeedbackPanel();
    setPending('live');
  }

  // Live feedback builder
  function buildFeedback() {
    const seconds = Math.round(xapi.totalPlayMs / 1000);
    const actionsHtml = actionLog.map(a => 't = ' + a.t + ', ' + a.message).join('<br>');
    return '<div><div>'
      + 'Interactions: ' + xapi.interactionCount + '<br>'
      + 'Total Play Time: ' + seconds + 's<br>'
      + 'Score: ' + (xapi.score || 0) + '<br><br>'
      + 'Action Log:<br>' + actionsHtml
      + '</div></div>';
  }

  // Floating live feedback panel
  const panel = document.createElement('div');
  Object.assign(panel.style, {
    position: 'fixed', right: '12px', bottom: '56px',
    minWidth: '240px', maxWidth: '320px',
    background: 'rgba(17,24,39,0.9)', color: '#e5e7eb',
    border: '1px solid #374151', borderRadius: '8px',
    padding: '8px 10px', zIndex: 999998, fontSize: '12px', lineHeight: '1.3'
  });
  panel.setAttribute('id', 'xapiLiveFeedback');
  document.addEventListener('DOMContentLoaded', () => {
    // Append panel near the end so the Save button remains on top
    try { document.body.appendChild(panel); } catch(e){}
    updateFeedbackPanel();
  });

  function updateFeedbackPanel() {
    try {
      const seconds = Math.round(xapi.totalPlayMs / 1000);
      panel.innerHTML = ''
        + '<div style="font-weight:600;margin-bottom:6px;">Live Feedback</div>'
        + '<div>Interactions: ' + xapi.interactionCount + '</div>'
        + '<div>Total Play Time: ' + seconds + 's</div>'
        + '<div>Score: ' + (xapi.score || 0) + '</div>';
    } catch(e){}
  }

  // Keep play time ticking in panel while playing
  setInterval(() => {
    if (xapi.playStart) updateFeedbackPanel();
  }, 1000);

  // Keep a pending payload up to date for auto-send triggers
  function setPending(reason) {
    try {
      const feedback = buildFeedback();
      window.__xapiPending = { score: xapi.score, feedback, reason, actions: actionLog };
    } catch(e){}
  }

  document.addEventListener('click', (e) => {
    const el = e.target;
    let label = '';
    if (el && el.tagName === 'BUTTON') {
      label = el.innerText.trim() || el.id || 'button';
      logAction('click "' + label + '"');
      xapi.interactionCount++;
    } else if (el && el.tagName === 'INPUT') {
      label = (el.value || el.name || el.id || 'input').toString();
      logAction('click input "' + label + '"');
      xapi.interactionCount++;
    }
  }, true);

  document.addEventListener('change', (e) => {
    const el = e.target;
    if (el && el.tagName === 'SELECT') {
      const val = el.value;
      const label = val ? (val.charAt(0).toUpperCase() + val.slice(1)) : 'Select';
      logAction('click "' + label + '"');
      xapi.interactionCount++;
    }
  }, true);

  let dragStart = null;
  document.addEventListener('mousedown', (e) => {
    if (e.target && e.target.tagName === 'CANVAS') {
      dragStart = { x: e.offsetX, y: e.offsetY, canvas: e.target };
    }
  }, true);
  document.addEventListener('mouseup', (e) => {
    if (dragStart && dragStart.canvas === e.target) {
      const name = 'canvas';
      logAction('drag ' + name + ' to position ' + Math.round(e.offsetX) + ',' + Math.round(e.offsetY) + '.');
      xapi.interactionCount++;
    }
    dragStart = null;
  }, true);

  function hookById(id, message, after) {
    const el = document.getElementById(id);
    if (!el) return;
    el.addEventListener('click', () => {
      logAction(message);
      if (typeof after === 'function') after();
    }, true);
  }

  hookById('playBtn', 'clicked "Play"', () => {
    if (!xapi.playStart) xapi.playStart = Date.now();
    setPending('play');
  });

  hookById('pauseBtn', 'click "Pause"', () => {
    flushPlayTime();
    setPending('pause'); // keep pending updated
    // Auto-save on pause remains as a convenience
    if (typeof window.storeState === 'function') {
      try {
        const feedback = buildFeedback();
        const payload = { score: xapi.score, feedback, reason: 'pause', actions: actionLog };
        window.storeState(payload);
        logAction('auto-save (pause)');
      } catch(e) { console.warn('xAPI sendState failed:', e); }
    }
  });

  hookById('resetBtn', 'click "Reset"', () => {
    xapi.score = (typeof xapi.score === 'number') ? xapi.score + 1 : 1;
    logAction('awarded 1 mark. total score = ' + xapi.score + '.');
    setPending('reset');
  });

  // Dedup-safe sender used by auto hooks (including SLS submit via postMessage)
  let lastSignature = null;
  function signatureOf(p) {
    try { return JSON.stringify({ s: p.score, a: (p.actions ? p.actions.length : 0), f: (p.feedback || '').length }); }
    catch { return String(Date.now()); }
  }
  function sendState(reason) {
    try {
      flushPlayTime();
      const feedback = buildFeedback();
      const payload = { score: xapi.score, feedback, reason, actions: actionLog };
      const sig = signatureOf(payload);
      if (sig && sig === lastSignature) return; // avoid duplicate identical sends
      lastSignature = sig;

      if (typeof window.storeState === 'function') {
        window.storeState(payload);
        logAction('auto-save (' + reason + ')');
      } else {
        console.warn('storeState not available; ensure xAPI scripts are loaded and platform URL params are present.');
      }
      // keep pending in sync
      setPending(reason);
    } catch(e) {
      console.warn('xAPI sendState failed:', e);
    }
  }

  // Floating "Save to SLS" button (kept as required user action)
  const btn = document.createElement('button');
  btn.textContent = 'Save to SLS';
  Object.assign(btn.style, {
    position: 'fixed', right: '12px', bottom: '12px',
    padding: '8px 12px', background: '#1976d2', color: '#fff',
    border: 'none', borderRadius: '6px', zIndex: 999999, cursor: 'pointer'
  });
  btn.addEventListener('click', () => {
    flushPlayTime();
    if (typeof window.storeState === 'function') {
      try {
        const feedback = buildFeedback();
        const payload = { score: xapi.score, feedback, reason: 'manual-save', actions: actionLog };
        window.storeState(payload);
        lastSignature = signatureOf(payload); // dedup against autos
        logAction('manual save triggered');
      } catch(e) { console.warn('xAPI sendState failed:', e); }
    } else {
      console.warn('storeState not available; ensure xAPI scripts are loaded and platform URL params are present.');
    }
    setPending('manual-save');
  });
  document.addEventListener('DOMContentLoaded', () => {
    try { document.body.appendChild(btn); } catch(e){}
  });

  // Auto-send hooks:
  // 1) When content is being hidden or navigated away
  document.addEventListener('visibilitychange', () => {
    if (document.visibilityState === 'hidden') sendState('visibility-hidden');
  }, true);
  window.addEventListener('pagehide', () => {
    sendState('pagehide');
  }, { capture: true });

  // 2) Listen for parent window messages indicating "submit" (SLS submit button)
  //    If SLS can postMessage({ type: 'submit' }) on submit, this will auto-send.
  window.addEventListener('message', (event) => {
    try {
      const data = event.data;
      const key = (data && (data.type || data.event || data.action || data.message)) || '';
      if (typeof key === 'string' && /submit/i.test(key)) {
        sendState('sls-submit');
      }
    } catch(e){}
  }, false);

  // Initialize pending once
  setPending('init');
})();
        `.trim();

        const s3 = doc.createElement('script');
        s3.textContent = snippet;
        doc.body.appendChild(s3);
      }
    }

    return serializeDocument(doc);
  }

  integrateBtn.addEventListener('click', async () => {
    setStatus('');
    downloadLink.style.display = 'none';
    downloadLink.removeAttribute('href');
    downloadLink.removeAttribute('download');
    if (logEl) { logEl.textContent = ''; }

    const file = (zipInput && zipInput.files && zipInput.files[0]) ? zipInput.files[0] : null;
    if (!file) {
      setStatus('Please choose a ZIP file to integrate.');
      return;
    }

    setStatus('Loading ZIP...');
    log(`Reading ZIP: ${file.name} (${file.size} bytes)`);

    try {
      if (!window.JSZip) {
        setStatus('JSZip failed to load. Check network or ensure ./vendor/jszip.min.js exists.');
        log('JSZip not available on window.');
        return;
      }
      const jszip = new JSZip();
      const zip = await jszip.loadAsync(file);

      const { contentDir, indexPath } = findContentDirFromZip(zip);
      if (!indexPath) {
        setStatus('index.html not found in ZIP.');
        log('Error: index.html not found.');
        return;
      }
      log(`Detected content dir: "${contentDir || '(root)'}", index: "${indexPath}"`);

      const html = await zip.file(indexPath).async('string');

      const mode = getSelectedMode();
      log(`Injecting (mode=${mode})...`);
      const injectedHtml = injectScriptsIntoHtml(html, { mode });

      // Write back modified index.html
      zip.file(indexPath, injectedHtml);

      // Load vendors from our public/vendor and add to ZIP under contentDir/lib/
      const vendors = await loadVendors();
      const libBase = joinPath(contentDir, 'lib/');
      zip.file(joinPath(libBase, 'xapiwrapper.min.js'), vendors.wrapper);
      zip.file(joinPath(libBase, 'xAPI.js'), vendors.glue);

      setStatus('Packaging ZIP...');
      const outBlob = await zip.generateAsync({ type: 'blob' });
      const baseName = file.name.replace(/\.zip$/i, '');
      const outName = `integrated_${baseName}.zip`;

      const url = URL.createObjectURL(outBlob);
      downloadLink.href = url;
      downloadLink.download = outName;
      downloadLink.style.display = 'inline-block';
      setStatus('Done. Click "Download ZIP".');
      log(`Generated output: ${outName}`);
    } catch (e) {
      console.error(e);
      setStatus('Integration failed. See log.');
      log(`Error: ${e && e.message ? e.message : String(e)}`);
    }
  });
})();
