/*
 * Requires Polynomial.js
 * Polynomial implements a mathematical polynomial:
 * c[0] + c[1] * x + c[2] * x^2 + ....
*/


function Laguerre() {
    this.assocLaguerreMap = new Map();
    this.laguerreList=new Array(0);
    // seed the first two Laguerre polynomials
    p0 = new Polynomial([1.0]);
    this.laguerreList.push(p0); 
    p1 = new Polynomial([1.0, -1.0]);
    this.laguerreList.push(p1);  
}

/**
 * Gets the n-th Laguerre polynomial. If the polynomial has already been calculated
 * it is returned  from the list. Uses the recurrence relationship to construct new polynomials
 * based on lower order polynomials.
*/
Laguerre.prototype.getPolynomial = function(n){
    if(n<0){
      console.log("Error n negative in Laguerre. n="+n);
      return null;
    }

	if(n<this.laguerreList.length) {
      //console.log("Has Laguerre: n="+n);
      return this.laguerreList[n];
    }
    var pk = new Polynomial( [(2.0*n-1.0)/n, -1.0/n]);
	var p1 = this.getPolynomial(n-1).multiply(pk);
	var p2 = this.getPolynomial(n-2).multiply((1.0-n)/n);  //scalar multiply
    var p = p1.add(p2);
    //console.log("Add Laguerre: n="+n);
    this.laguerreList.push(p); // polynomial was not on the list so add it.
    return p;
}

/**
 * Gets the associated Laguerre polynomial. If the polynomial has already been calculated
 * it is returned  from the list. Uses the recurrence relationship to construct new polynomials
 * based on lower order polynomials.
 */
Laguerre.prototype.getAssocPolynomial = function(n, k){
    if(k<0){
        console.log("Error k negative in associated Laguerre. k="+k);
        return null;
    } 
    let key=[n,k];
    if(this.assocLaguerreMap.has(key)) {
      //console.log("Has assoc Laguerre: key="+key);
      return this.assocLaguerreMap.get(key);;
    }
    p = this.getPolynomial(n+k);
    let sign = 1;
    for(let i = 0; i<k; i++) {
      sign *= -1;
      p = p.derivative();
    }
    if(sign==-1) {
      p = p.multiply(sign);
    }
    //console.log("Add assoc Laguerre: key="+key);
    this.assocLaguerreMap.set(key, p); // polynomial was not in the list so add it.
    return p;
}